	function [h, err] = RERMEZ_FIR(order, edge, fx, varargin)
	
	% RERMEZ_FIR - A translation of the FORTRAN code of the Parks-McClellan
	% minimax arbitrary-magnitude FIR filter design algorithm [1], referred 
	% in later use to as the PMR code, into MATLAB.
	
	% 	Toolbox for DIGITAL FILTERS USING MATLAB
	
	% 	Author: 		Tapio Saramaki, 2018-02-20
	% 	Modified by: 	
	% 	Copyright:		by authors - not released for commercial use
	% 	Version:		1 
	% 	Known bugs:		Do error checks!
	% 	Report bugs to:	tapio.saramaki@tut.fi
	
	
	% Compared with the PMR code, the the peculiarities of the present code are: 
	% (1) The Remez loop has been improved according to [2].
	% (2) Due to the use of MATLAB, the grid pont allocation happened to be 
	%     improved.
	% (3) The desired function is given in a slightly different manner.
	%
	% Inputs to be always used:
	% order - filter order
	% edge  - specifies the upper and lower edges of the bands under consideration.
	%		  The program, however, uses band edes in terms of fractions of pi rad.
	edge = edge/pi;
	% fx -    specifies the desired values at the edges of each band.
	%
	% Inputs to a possible inclusion in varargin: 
	% wtx   - a constant weighting in each band
	% type  - three filter types are under consideration, namely,
	%         (1) Filters with symmetric impulse response for multiband design: 
	%             lowpass, higpass, bandpass, and bandstop designs are typical
	%             examples. The benefit of these filters is that the phase is
	%             perfectly linear for all frequencies.
	%         (2) Full band and partial band differentiators: filters with 
	%             antisymmetric impulse response are in use.
	%         (3) Hilbert transformers: again antisymmetry is required.
	%         The distinction between these three types is performed by a 
	%         character array, called type, such that
	%         type(1)='m' or type(1)='M' ==> 'multiband' design
	%         type(1)='d' or type(1)='D' ==> 'differentiator'
	%         type(1)='h' or type(1)='H' ==> 'Hilbert' tranformer
	%         If type is not specified, then type=='multiband' is used
	% lgrid - if not specified, lgrid=16 as in the PMR code. To  increase the 
	%         accuray of the resulting filter, lgrid = 32, lgrid = 64, 
	%         lgrid=128,... is recommended. lgrid should be specified as {32},
	%         {64}, {128},... in order to easily find its existance.
	% It is assumed that if wtx, type, and lgrid or two out of them are in use, 
	% then they are specified in the above order.
	%--------------------------------------------------------------------------
	% Compared with the PMR code, there are two differences in the input data:
	% First, in the PMR code, lgrid is specified inside the code and, secondly,
	% in the present code, the desired function, instead of being a constant as
	% in the PMR code, is given at both band edges meaning that the desired 
	% function is a straight line passing two x-y coordinates. This option has 
	% been introduced after following the trials of some MATLAB people to 
	% translate the PMR code into MATLAB. Perhaps, those MATLAB people could not
	% understand that for differentiators the desired function is a straight 
	% line. Anyway, this option is now in use!
	%------------------------------------------------------------------------- 
	%	Example of a lowpass filter of orddeer 30, i.e., length 31
	%   	[h, Err] =  EQR_FIR(30,[0 0.2 0.4 1]*pi,[1 1 0 0],[1 10], 'm');
	%
	%	Example of a   differentiator
	%   	[hodd, Err] = REMEZ_FIR(31,[0 1]*pi,[1 1],  'd');
	%		[heven, Err] = REMEZ_FIR(30,[0 0.9*pi],[1 1], 'd');
	%
	%	Example of a Hilbert filter
	%		[h, Err] = REMEZ_FIR18, [0.1 0.9]*pi, [1 1], 'h');
	%------------------------------------------------------------------------- 
	%
	% Outputs
	%     h - coefficients of the filter
	%     err - the final absolute value of the weighted error function
	%
	% References:
	% [1] J. H. McClellan, T. W. Parks, and L. R. Rabiner, "A computer program 
	%     for designing optimum FIR linear phase digital filters," IEEE 
	%     Transactions on Audio and Electroacoustics, vol. AU-21, no. 6, pp.
	%     506 - 526, December 1973; 
	%  -  also reprinted in IEEE Acoustics, Speech, and Signal Processing 
	%     Society. Digital Signal Processing Committee, eds., "Selected 
	%     Papers in Digital Signal Processing, II", pp. 97 - 117, IEEE Press, 
	%     New York, 1976; 
	%  -  the FORTRAN program itself and its short description is included in 
	%     IEEE Acoustics, Speech, and Signal Processing  Society. Digital 
	%     Signal Processing Committee, eds., "Programs for Digital Signal 
	%     Processing", pp. 5.1-1 - 5.1-13, IEEE Press, John Wiley & Sons 
	%     New York, 1979.
	% [2] M. Ahsan and T. Saramki, "A MATLAB based optimum multiband FIR  
	%      filters design program following the original idea of the Remez 
	%      multiple exchange algorithm," in Proc. 2011 IEEE International 
	%      Symposium on Circuits and Systems, Rio de Janeiro, Brazil, 
	%      May 1517, 2011, pp. 137  140. 
	%
	%==========================================================================
	% Check varargin according to the above assumptions
	%==========================================================================
	lgrid = 16; % default value
	nn = length(varargin);
	if nn > 0 & iscell(varargin{nn})
	    lgrid = varargin{nn}{:};
	    nn = nn-1;
	    varargin = varargin(1:nn);
	end
	if nn==1
	    if ischar(varargin{1})
	        type = varargin{1};
	        wtx = ones(1,length(fx)/2);
		else
	        wtx = varargin{1};
	        type = 'multiband';
		end
	elseif nn==2
	    wtx = varargin{1};
	    type = varargin{2};
	else
	    wtx=ones(1, length(fx)/2);
	    type = 'multiband';
	end
	%==========================================================================
	%  Find out jtype that was used in the PM code.
	%  This not necessary but simplifies the undertanding of this code snippet.
	%==========================================================================
	if type(1) == 'd' | type(1) == 'D'
		jtype = 2;	% Differentiator
	elseif type(1) == 'h' | type(1) == 'H'
		jtype = 3;	% Hilbert transformer 
	elseif type(1) == 'm' | type(1) == 'M'
		jtype = 1;	% Multiband filter
	else
		jtype = 1;	% Multiband filter
	end
	%==========================================================================
	% Determine the filter cases and nfcns, the number of basis functions to be 
	% used in the Remez algorithm 
	% In the below, filtercase=1,2,3,4 is used for making it easier to 
	% understand this code snippet.   
	%==========================================================================
	if jtype==1
	    if rem(order,2) == 0  
	        filtercase = 1; % Even order and even symmetry; multiband filter
		else
	        filtercase = 2; % Odd order and even symmetry; multiband filter
		end
	else
	    if rem(order,2) == 0
	        filtercase = 3; % Even order and odd symmetry; a Hilbert transfor-
			% er or a differentiator (jtype indicates)
		else
	        filtercase = 4; % Odd order and odd symmetry; a Hilbert transformer
			% or a differentiator (jtype indicates)
		end
	end
	nfcns = fix((order+1)/2);
	if filtercase == 1
	    nfcns = nfcns + 1;
	end
	%  ===>
	% nfcns = order/2+1 for filtercase = 1    	no fixed zeros
	% nfcns = (order+1)/2 for filtercase = 2; 	fixed zero at z = -1 ==> 
	%                                     		fixed term (1+z^{-1})/2
	% nfcns = order/2 for filtercase = 3;     	fixed zeros at z = 1 and z = -1 ==> 
	%                                     		fixed term (1-z^{-2})/2
	% nfcns = (order+1) for filtercase = 4;   	fixed zero at z = -1 ==> 
	%                                    		fixed term (1-z^{-2})/2
	%=========================================================================
	% DETERMINE grid, des, and wt 
	%========================================================================
	% Compared with the PM code, there are the following key differences:
	% (1) The upper edge for each band under consideration is automatically 
	%     included in grid. This somehow increases the accuracy. 
	% (2) Since the frequency range is now from 0 to 1, delf has been increased
	%     by a factor of 2.
	% (3) The change of des and wt depending on the filter type is peformed 
	%     before using the (modified) Remez algorithm.
	% (4) The removal of problematic angular frequencies at 0 and pi is 
	%     performed simultaneously for all filter types. Now the remomal is
	%     is performed while generating grid.
	%=========================================================================
	nbands = length(edge)/2;
	delf = 0.5/(lgrid*nfcns); 
	delf = 2*delf;
	grid = []; des = []; wt = [];
	ll=1;
	while ll <= nbands
	    number_grid = ceil((edge(2*ll)-edge(2*ll-1))/delf);
	    grid_more=linspace(edge(2*ll-1),edge(2*ll),number_grid+1);
	    
	    % TEST whether there exists harmful omega = 0 or omega = pi or both
	    
	    if ll==1 &(filtercase == 3 | filtercase==4) & grid_more(1)< delf
	        grid_more(1) = [];
	        number_grid = number_grid-1;
		end
	
	    if ll==nbands & (filtercase==2 | filtercase==3) & grid_more(end)>1-delf
	        grid_more(end)=[];
	        number_grid=number_grid-1;
		end
		grid = [grid grid_more]; 
	    if jtype~=2
	        wt_more = wtx(ll)*ones(1,number_grid+1); 
	        if fx(2*ll)~= fx(2*ll-1) 
	            des_more = linspace(fx(2*ll-1),fx(2*ll),number_grid+1);
			else
	            des_more = fx(2*ll)*ones(1,number_grid+1);
			end
		else
	        des_more = fx(2*ll)*grid_more*pi;
	        if abs(fx(2*ll-1)) < 1.0e-3
	            wt_more = wtx(ll)*ones(1,number_grid+1);
			else
	            wt_more = wtx(ll)./(grid_more*pi);
			end      
		end
	    des = [des des_more];
	    wt = [wt wt_more];
	    ll = ll+1;
	end
	%========================================================================
	% Compared the PM code, there are two basic differences:
	% (1) In the allocation of grid points in each band, delf_new has been 
	%     predetermined such that the last point automatically occurs at the 
	%     upper edge. In the original code, it becomes, due to the use of delf, 
	%     in most cases larger than the edge. As a consequence, the last point
	%     is removed and the value of previous grid point is fixed to take on
	%     the edge value. Hence, the distance within the last and second last
	%     grid points is in most cases larger than delf.
	% (2) In the jtype=2 case, des=pi*grid*fx, instead of des=grid*fx, and
	%     wt=wtx/(pi*grid), instead of wt=wtx/grid, are used provided that 
	%     fx >= 1.0e-3.
	%==========================================================================
	% Modify des and wt depending on the filter case
	%========================================================================== 
	if filtercase == 2
	    des = des./cos(pi*grid/2); % The response of the fixed term 
	    wt = wt.*cos(pi*grid/2);   % (1+z^{-1})/2 is cos(omega/2)
	end
	if filtercase == 4
	    des = des./sin(pi*grid/2); % The response of the fixed term 
	    wt = wt.*sin(pi*grid/2);   % (1+z^{-1})/2 is sin(omega/2)
	end
	if filtercase == 3
	    des = des./sin(pi*grid);   % The response of the fixed term
	    wt = wt.*sin(pi*grid);     % (1+z^{-2})/2 is sin(omega)
	end
	%==========================================================================
	% CALL THE REMEZ ALGORITHM 
	%==========================================================================
	% Compared with the PM code, there are the following key differences:
	% (1) On purpose, only filters with even order and even symmetry are under
	%     consideration or filtercase = 1 is in use. As a matter of fact, the
	%     original FORTAN code did it indirectly.
	% (2) For achieving this goal, des and wt modified beforehand and the 
	%     possible response for another filter type will be generated later 
	%     based on the present response
	% (3) The Remez multiple exchange loop has been significantly improved.
	% (4) grid, the frequency grid, is now within 0 and 1, instead of being
	%     within 0 and 0.5.
	%========================================================================== 
	[h, err,iext] = REMEZ_EX_A(nfcns, grid, des, wt);
	%== =======================================================================
	% Generate the impulse responses for other types 
	%=======================================================================
	nn = length(h);
	if filtercase == 2
	    h = [h(1)/2 (h(2:nn)+h(1:nn-1))/2 h(nn)/2]; 
	end
	if filtercase == 3
	    h = [h(1)/2 h(2)/2 (h(3:nn)-h(1:nn-2))/2 -h(nn-1)/2 -h(nn)/2]; 
	end
	if filtercase == 4
	    h = [h(1)/2 (h(2:nn)-h(1:nn-1))/2 -h(nn)/2]; 
	end
	err = abs(err);
	%%
	
	
